import React, { useState, useEffect } from "react";
import axios from 'axios';
import Form from "@rjsf/core";
import validator from "@rjsf/validator-ajv8";

// JSON schema for Pricing Rules Form
export const userSchema = {
  title: "Service Form",
  description: "Create or update Service.",
  type: "object",
  required: [
    "service_category",
    "service_title",
    "service_description",
    "capacity_jobs",
    "has_cost",
    "is_vehicle_focused",
    // "image",
    // "map_icon",
  ],
  properties: {
    service_category: {
      type: "string",
      title: "Service Category",
      oneOf: [
        // Default empty option
        {
          const: "",
          title: "Select Category",
        },
        // Dynamic options will be added after this
      ],
    },
    service_title: {
      type: "string",
      title: "Service Name",
    },
    service_description: {
      type: "string",
      title: "Service Description",
    },
    skills: {
      type: "array",
      title: "Skills",
      items: {
        type: "string",
        oneOf: [], // Will be populated dynamically
      },
      uniqueItems: true,
    },
    capacity_jobs: {
      type: "number",
      title: "Capacity",
    },
    has_cost: {
      type: "number", // Change to enum with 1 and 0 for radio
      title: "Has Cost",
      enum: [1, 0],
      enumNames: ["Yes", "No"],
    },
    is_vehicle_focused: {
      type: "number", // Change to enum with 1 and 0 for radio
      title: "Is Vehicle Focused",
      enum: [1, 0],
      enumNames: ["Yes", "No"],
    },    
    checklist: {
      type: "array",
      title: "Checklist",
      items: {
        type: "string",
        oneOf: [], // Will be populated dynamically
      },
      uniqueItems: true,
    },
    addons: {
      type: "array",
      title: "Addons",
      items: {
        type: "string",
        oneOf: [], // Will be populated dynamically
      },
      uniqueItems: true,
    },
    image: {
      type: "string",
      title: "Service Image",
      format: "data-url",
    },
    map_icon: {
      type: "string",
      title: "Map Icon",
      format: "data-url",
    },
    account_id: {
      type: "string", // Use string or number based on your DB
      title: "User ID",
      default: "{id}",
    },
  },
};

const CustomFileWidget = (props) => {
  const { value, onChange } = props;

  const handleChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (event) => {
        onChange(event.target.result);
      };
      reader.readAsDataURL(file);
    }
  };

  return (
    <div>
      {value && (
        <div>
          <img 
            src={value} 
            alt="Preview" 
            style={{ maxWidth: '100px', marginBottom: '10px' }} 
          />
          <p>Current: {value.startsWith('data:') ? 'New Upload' : value}</p>
        </div>
      )}
      <input 
        type="file" 
        onChange={handleChange} 
        accept="image/*" 
      />
    </div>
  );
};

// Dynamic Fields
const DynamicFieldManager = ({ fields, setFields }) => {
  const addField = () => {
    setFields([...fields, { label: '', type: 'text', options: '' }]);
  };

  const removeField = (index) => {
    const newFields = fields.filter((_, i) => i !== index);
    setFields(newFields);
  };

  const updateField = (index, key, value) => {
    const newFields = [...fields];
    newFields[index][key] = value;
    setFields(newFields);
  };

// Check if all existing fields are filled
  const canAddNew = fields.every(f => f.label.trim() && f.type && 
    (['select', 'checkbox'].includes(f.type) ? f.options.trim() : true));

  return (
    <div className="dynamic-fields">
      {fields.map((field, index) => (
        <div key={index} className="field-row" style={{ marginBottom: '1rem' }}>
          <input
            type="text"
            placeholder="Field Label"
            value={field.label}
            onChange={(e) => updateField(index, 'label', e.target.value)}
            style={{ marginRight: '1rem', width: '150px' }}
          />
          
          <select
            value={field.type}
            onChange={(e) => updateField(index, 'type', e.target.value)}
            style={{ marginRight: '1rem', width: '120px' }}
          >
            <option value="text">Text</option>
            <option value="textarea">Textarea</option>
            <option value="select">Select</option>
            <option value="checkbox">Checkbox</option>
          </select>

          {['select', 'checkbox'].includes(field.type) && (
            <input
              type="text"
              placeholder="Options (comma separated)"
              value={field.options}
              onChange={(e) => updateField(index, 'options', e.target.value)}
              style={{ marginRight: '1rem', width: '200px' }}
            />
          )}

          <button
            type="button"
            onClick={() => removeField(index)}
            style={{ marginLeft: '0.5rem' }}
          >
            -
          </button>
        </div>
      ))}

      <button
        type="button"
        onClick={addField}
        disabled={!canAddNew}
        style={{ marginTop: '1rem' }}
      >
        Add Field +
      </button>
    </div>
  );
};

// Optional UI schema tweaks
export const userUiSchema = {
  image: {
    "ui:widget": "file", // File upload for images
  },
  map_icon: {
    "ui:widget": "file", // File upload for map icons
  },
  is_vehicle_focused: {
    "ui:widget": "radio", // Set as radio button
  },
  has_cost: {
    "ui:widget": "radio", // Set as radio button
  },
  user_id: {
    "ui:widget": "hidden", // Hide user_id field
  },
  account_id: {
    "ui:widget": "hidden", // Hide user_id field
  },
  checklist: {
    "ui:widget": "select",
    "ui:options": {
      multiple: true,
      // Optional: Add placeholder text
      placeholder: "Select checklists...",
    },
  },
  skills: {
    "ui:widget": "select",
    "ui:options": {
      multiple: true,
      // Optional: Add placeholder text
      placeholder: "Select skills...",
    },
  },
  addons: {
    "ui:widget": "select",
    "ui:options": {
      multiple: true,
      // Optional: Add placeholder text
      placeholder: "Select addons...",
    },
  },
  "ui:submitButtonOptions": {
    norender: true, // Hide default submit button
  },
};

const UserForm = ({ initialData, onSubmit }) => {
  const [formData, setFormData] = useState(initialData || {});
  const [schema, setSchema] = useState(userSchema);
  const [dynamicFields, setDynamicFields] = useState([]);
  const formRef = React.useRef();

  
  useEffect(() => {
    const fetchChecklist = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getchecklist/${userId}`);

        if (response.data?.users) {
          const checkList = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.checklist_name,
          }));

          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              checklist: {
                ...prev.properties.checklist,
                items: {
                  ...prev.properties.checklist.items,
                  oneOf: checkList,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching checklists:", error);
      }
    };

    fetchChecklist();
  }, []);

  useEffect(() => {
    const fetchSkills = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getskills/${userId}`);

        if (response.data?.users) {
          const skills = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.name,
          }));

          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              skills: {
                ...prev.properties.skills,
                items: {
                  ...prev.properties.skills.items,
                  oneOf: skills,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching skills:", error);
      }
    };

    fetchSkills();
  }, []);

  useEffect(() => {
    const fetchAddons = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getaddonlist/${userId}`);

        if (response.data?.users) {
          const addons = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.name, // Matches API's "name" field
          }));

          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              addons: {
                ...prev.properties.addons,
                items: {
                  ...prev.properties.addons.items,
                  oneOf: addons,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching addons:", error);
      }
    };

    fetchAddons();
  }, []);


  //**** For Get Service Category List ****************//
  // In the ServicesForm.js component
  useEffect(() => {
      const fetchServices = async () => {
        try {
          const userId = localStorage.getItem('userId');
          const response = await axios.get(`/services/getservicecategorylist/${userId}`);

          if (response.data?.users) {
            const serviceOptions = [
              // Keep the default empty option
              ...userSchema.properties.service_category.oneOf,
              // Add API options
              ...response.data.users.map((service) => ({
                const: service.user_id,
                title: service.service_category,
              }))
            ];

            setSchema(prev => ({
              ...prev,
              properties: {
                ...prev.properties,
                service_category: {
                  ...prev.properties.service_category,
                  oneOf: serviceOptions,
                }
              }
            }));
          }
        } catch (error) {
          console.error("Error fetching services:", error);
        }
      };

      fetchServices();
    }, []);


  /*const handleSubmit = ({ formData }) => {
    console.log("Form Submitted:", formData);
    onSubmit(formData);
  };*/

  /*const handleSubmit = ({ formData }) => {
    const submissionData = {
      ...formData,
      dynamicFields: dynamicFields.map(f => ({
        ...f,
        options: f.options ? f.options.split(',').map(o => o.trim()) : []
      }))
    };
    console.log("Form Submitted:", submissionData);
    onSubmit(submissionData);
  };*/

  const handleSubmit = ({ formData }) => {
    const submissionData = {
      ...formData,
      dynamicFields: JSON.stringify(dynamicFields.map(f => ({
        ...f,
        options: f.options ? f.options.split(',').map(o => o.trim()) : []
      })))
    };
    onSubmit(submissionData);
  };

  return (
    <div style={{ maxWidth: "600px", margin: "0 auto" }}>
      <Form
        ref={formRef}
        schema={schema}
        uiSchema={userUiSchema}
        formData={formData}
        onChange={({ formData }) => setFormData(formData)}
        onSubmit={handleSubmit}
        validator={validator}
        liveValidate
        widgets={{ FileWidget: CustomFileWidget }}
      />
      
      <h3>Additional Fields</h3>
      <DynamicFieldManager 
        fields={dynamicFields} 
        setFields={setDynamicFields} 
      />

      {/* Custom submit button placed after additional fields */}
      <div style={{ marginTop: '20px', textAlign: 'right' }}>
        <button 
          type="button"
          onClick={() => formRef.current.submit()}
          style={{
            padding: '8px 20px',
            backgroundColor: '#007bff',
            color: 'white',
            border: 'none',
            borderRadius: '4px',
            cursor: 'pointer'
          }}
        >
          Submit
        </button>
      </div>
    </div>
  );
};

export default UserForm;
